<?php namespace Intellex\MailChimp\Api;

use Intellex\Curly\Method;
use Intellex\MailChimp\Exception\ExceptionFactory;
use Intellex\MailChimp\Resource\ResourceList;
use Intellex\MailChimp\Resource\ResourceSubscriber;
use Intellex\Pixabay\Validation\ValidationException;

/**
 * Class ListApi
 *  Handles all subscription list API operations.
 *
 * @package Intellex\MailChimp\Api
 */
class ApiLists extends ApiAbstract {

	/**
	 * @param array  $data   to create new list. Below is list of required fields
	 *                       -name,
	 *                       -contact[ (Contact information displayed in campaign footers to comply with international spam laws.)
	 *                       ----company (The company name for the list.)
	 *                       ----address1 (The street address for the list contact.)
	 *                       ----city (The city for the list contact.)
	 *                       ----state (The state for the list contact.)
	 *                       ----zip (The postal or zip code for the list contact.)
	 *                       ----country (A two-character ISO3166 country code. Defaults to US if invalid.)
	 *                       ],
	 *                       -permission_reminder (A permission reminder in your email campaign makes it clear where your recipients signed up for your list. ),
	 *                       -campaign_defaults [ (Default values for campaigns created for this list.)
	 *                       ----from_name (The default from name for campaigns sent to this list.)
	 *                       ----from_email (The default from email for campaigns sent to this list.)
	 *                       ----subject (The default subject line for campaigns sent to this list.)
	 *                       ----language (The default language for this lists’s forms.)
	 *                       ],
	 *                       -email_type_option => false (When set to true, subscribers can choose whether they want to receive HTML or plain-text emails. When set to false, subscribers will receive HTML emails, with a plain-text alternative backup.)
	 * @param string $listId null if creating new list, string if editing existing one.
	 *
	 * @throws \Exception|ValidationException
	 * @return mixed
	 */
	public function create($data, $listId = null) {
		$response = $this->curl([ 'lists', $listId ], $data, $listId ? Method::PATCH : Method::POST);
		return new ResourceList($response->getContent());
	}

	/**
	 * Get all of the list for the account.
	 *
	 * @param integer $offset Offset for the subscribers to show.
	 * @param integer $count  The number of subscribers to show.
	 *
	 * @return ResourceList[] The list of found lists.
	 */
	public function getAll($offset = 0, $count = 100) {
		$lists = $this->curl([ 'lists', [ $offset, $count ] ])->getContent();

		// Convert
		foreach ($lists['lists'] as $i => $list) {
			$lists['lists'][$i] = new ResourceList($list);
		}

		return $lists;
	}

	/**
	 * Get a list by its unique ID.
	 *
	 * @param string $listId The unique ID of the list.
	 *e
	 * @return ResourceList The found list, or null if not found.
	 */
	public function get($listId) {
		$data = $this->curl([ 'lists', $listId ])->getContent();

		return new ResourceList($data);
	}


	public function delete($listId) {
		return $this->curl([ 'lists', $listId ], '', Method::DELETE);
	}

	/**
	 * Get all of the list for the account.
	 *
	 * @param string  $listId The unique ID of the list.
	 * @param integer $offset Offset for the subscribers to show.
	 * @param integer $count  The number of subscribers to show.
	 *
	 * @return ResourceList[] The list of found lists.
	 */
	public function getSubscribers($listId, $offset = 0, $count = 100) {
		$response = $this->curl([ 'lists', $listId, 'members', [ $offset, $count ] ])->getContent();

		// Convert
		$subscribers = [];
		foreach ($response['members'] as $member) {
			$subscribers['members'][] = new ResourceSubscriber($member);
		}

		$subscribers['total_items'] = $response['total_items'];

		return $subscribers;
	}

}